//+------------------------------------------------------------------+
//|                                           Seasonal Decompose.mqh |
//|                                          Copyright 2023, Omegafx |
//|                 https://www.mql5.com/en/users/omegajoctan/seller |
//+------------------------------------------------------------------+
#property copyright "Copyright 2023, Omegafx"
#property link      "https://www.mql5.com/en/users/omegajoctan/seller"
//+------------------------------------------------------------------+
//| defines                                                          |
//+------------------------------------------------------------------+

vector moving_average(const vector &v, uint k, ENUM_VECTOR_CONVOLVE mode=VECTOR_CONVOLVE_VALID)
 {
   vector kernel = vector::Ones(k) /  k;
   vector ma = v.Convolve(kernel, mode);
    
   return ma;  
 }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
enum seasonal_model
 {
   additive,
   multiplicative
 };

struct seasonal_decompose_results
 {
   vector trend;
   vector seasonal;
   vector residuals;
 };
 
seasonal_decompose_results seasonal_decompose(const vector &timeseries, uint period, seasonal_model model=additive)
 {
   seasonal_decompose_results res;
   
   if (timeseries.Size() < period)
    {
      printf("%s Error: Time series length is smaller than the period. Cannot compute seasonal decomposition.",__FUNCTION__);
      return res;
    }
   
//--- compute the trend

   int n = (int)timeseries.Size();  
   res.trend = moving_average(timeseries, period);
   
   // We align trend array with the original series length
   
   int pad = (int)MathFloor((n - res.trend.Size()) / 2.0);
   int pad_array[] = {pad, n-(int)res.trend.Size()-pad};
   
   res.trend = Pad(res.trend, pad_array, edge);

//--- compute the seasonal component
   
   if (model == multiplicative)
     {
       for (ulong i=0; i<timeseries.Size(); i++)
         if (timeseries[i]<=0)
            {
               printf("Error, Multiplicative seasonality is not appropriate for zero and negative values");
               return res;
            }
     }
   
   vector detrended = {};
   vector seasonal = {};
   
   switch(model)
     {
      case  additive:
        {
          detrended = timeseries - res.trend;
          seasonal = vector::Zeros(period);
         
          for (uint i = 0; i < period; i++)
            seasonal[i] = SliceStep(detrended, i, period).Mean(); //Arithmetic mean over cycles
        }    
        break;
      case  multiplicative:
        {
          detrended = timeseries / res.trend;
          seasonal = vector::Zeros(period);
         
          for (uint i = 0; i < period; i++)
            seasonal[i] = MathExp(MathLog(SliceStep(detrended, i, period)).Mean()); //Geometric mean
        }    
        break;
      default:
        printf("Unknown model for seasonal component calculations");
        break;
     }


    
    vector seasonal_repeated = Tile(seasonal, (int)MathFloor(n/period)+1);
    res.seasonal = Slice(seasonal_repeated, 0, n);

//--- Compute Residuals

    if (model == additive)
        res.residuals = timeseries - res.trend - res.seasonal;
    else  // Multiplicative
        res.residuals = timeseries / (res.trend * res.seasonal);
        
   return res;
 }
//+------------------------------------------------------------------+
//|     This function adds/pads (extra values) around an array       |
//+------------------------------------------------------------------+
enum pad_mode
 {
   constant,
   edge,
   reflect,
   symmetric
 };
 
vector Pad(const vector &v, int &pad_width[], pad_mode mode=constant, int constant_values=0)
 {
   if (pad_width.Size()!=2)
      {   
         printf("pad_width array must have exactly two elements");
         return vector::Zeros(0);
      }    
      
   int left_pad = pad_width[0], right_pad = pad_width[1];
   vector left_values = {}, right_values = {};
   
   switch(mode)
     {
      case  constant:
        
        left_values = full(left_pad, constant_values);
        right_values = full(right_pad, constant_values);
        
        break;
      case  edge:
        
        left_values = full(left_pad, v[0]);
        right_values = full(right_pad, v[v.Size()-1]);
        
        break;
      case  reflect:
        
        if (left_pad==0)
          left_values = vector::Zeros(0);
        else
          {
           vector sliced = Slice(v, 1, left_pad+1);
           left_values = Reverse(sliced);
          }
          
        if (right_pad==0)
          right_values = vector::Zeros(0);
        else
          {
           vector sliced = Slice(v,  -right_pad-1, -1);
           right_values = Reverse(sliced);
          }  
          
        break;
      case  symmetric:
        
        if (left_pad==0)
          left_values = vector::Zeros(0);
        else
          {
           vector sliced = Slice(v, 0, left_pad);
           left_values = Reverse(sliced);
          }
        if (right_pad==0)
          right_values = vector::Zeros(0);
        else
          {
           vector sliced = Slice(v, -right_pad, -1);
           right_values = Reverse(sliced);
          }
          
        break;
      default:
        printf("Unkown/Unsupported padding mode");
        break;
     }
   
   return Concatenate(left_values, v, right_values);
 }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
template <typename T>
vector full(uint size, T fill_value)
 {
   vector res = vector::Zeros(size);
   res.Fill(fill_value);
   
   return res; 
 }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
vector Slice(const vector &v, int start_index, int end_index)
{
   int n = (int)v.Size();

   // Handle negative indexing (Python-style)
   if (start_index < 0) 
      start_index += n;
   
   if (end_index < 0)
      end_index += n;
   
   // Ensure valid indices
   if (start_index < 0 || start_index >= n || end_index < 0 || end_index > n)
   {
      printf("%s Failed: Index out of range", __FUNCTION__);
      return v;
   }

   if (start_index >= end_index)
   {
      printf("%s Failed: Start index >= End index, returning empty vector", __FUNCTION__);
      return v;
   }

   // Ensure correct sizing
   int size = end_index - start_index;
   vector res;
   res.Resize(size);

   for (int i = start_index, count = 0; i < end_index; i++, count++)
      res[count] = v[i];

   return res;
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
vector Reverse(const vector &v)
{
   vector v_reversed;
   v_reversed.Resize(v.Size()); // Allocate space for the reversed vector

   for (ulong i = 0, j = v.Size() - 1; i < v.Size(); i++, j--)
      v_reversed[i] = v[j];  // Copy elements in reverse order

   return v_reversed; // Return the new reversed vector
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
vector Concatenate(const vector &v1, const vector &v2, const vector &v3)
{
   vector result = vector::Zeros(v1.Size() + v2.Size() + v3.Size()); // Allocate space for the merged vector

   int index = 0;
   
   // Copy first vector
   for (ulong i = 0; i < v1.Size(); i++, index++)
      result[index] = v1[i];

   // Copy second vector
   for (ulong i = 0; i < v2.Size(); i++, index++)
      result[index] = v2[i];

   // Copy third vector
   for (ulong i = 0; i < v3.Size(); i++, index++)
      result[index] = v3[i];

   return result; // Return the concatenated vector
}
//+------------------------------------------------------------------+
//|  Constructs a large vector by repeating the vector v by a given  |
//|  number of reps value                                            |
//+------------------------------------------------------------------+
vector Tile(const vector &v, uint reps)
{
   if (reps <= 0)
    {
      printf("Error: Tile reps must be > 0. Returning empty vector.");
      return vector::Zeros(0);
    }
//---

   uint v_size = (uint)v.Size();
   uint res_size = v_size * reps;
   vector res = vector::Zeros(res_size);
   
   for (uint i = 0; i < reps; i++)
      for (uint j = 0; j < v_size; j++)
         res[i * v_size + j] = v[j];
   
   return res; 
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
vector SliceStep(const vector &v, int start, int step)
{
   int n = (int)v.Size();
   // If start is out of bounds, return an empty vector
   if(start < 0 || start >= n)
      return vector::Zeros(0);
      
   // Calculate the number of elements we'll extract
   int count = (n - start + step - 1) / step;  // ceiling division
   vector res = vector::Zeros(count);
   
   int idx = 0;
   for(int i = start; i < n; i += step)
   {
      res[idx] = v[i];
      idx++;
   }
   
   return res;
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
